<?php
/*
Plugin Name: FraudBD
Description: Adds a fraudBD fraud check to WooCommerce order view and manual search in admin menu.
Version: 1.1
Author: FraudBD
License: GPLv2 or later
Text Domain: fraudbd
Plugin URI: https://fraudbd.com/
*/

if (!defined('ABSPATH')) exit;

// Include settings and manual search page
require_once plugin_dir_path(__FILE__) . 'includes/settings-page.php';
require_once plugin_dir_path(__FILE__) . 'includes/manual-search-page.php';

// Add settings link
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
    $settings_link = '<a href="admin.php?page=fraud-check-settings">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
});

// Add fraud check button in WooCommerce order view
add_action('woocommerce_admin_order_data_after_billing_address', function ($order) {
    if (!current_user_can('manage_options')) return;

    $api_key = get_option('fraud_check_api_key');
    $username = get_option('fraud_check_username');
    $password = get_option('fraud_check_password');

    if (!$api_key || !$username || !$password) {
        echo '<div class="fraud-alert fraud-alert-error">' .
            esc_html__('FraudBD API credentials missing.', 'fraudbd') . ' ' .
            '<a href="' . esc_url(admin_url('admin.php?page=fraud-check-settings')) . '">' . esc_html__('Set them here', 'fraudbd') . '</a>' .
            '</div>';
        return;
    }

    $phone = $order->get_billing_phone();
    $cleaned = preg_replace('/^\+?88/', '', $phone);

    if (!preg_match('/^01[3-9][0-9]{8}$/', $cleaned)) {
        echo '<div class="fraud-alert fraud-alert-warning">Invalid Bangladeshi phone number format.</div>';
        return;
    }

    ?>
    <style>
        .fraud-alert {
            padding: 12px;
            margin: 10px 0;
            border-radius: 4px;
            font-weight: 500;
        }
        .fraud-alert-error {
            background-color: #fdecea;
            color: #d63638;
            border-left: 4px solid #d63638;
        }
        .fraud-alert-warning {
            background-color: #fff8e5;
            color: #dba617;
            border-left: 4px solid #dba617;
        }
        .fraud-check-container {
            margin: 15px 0;
        }
        #fraud-check-result {
            margin-top: 20px;
            background: #f6f7f7;
            padding: 20px;
            border: 1px solid #dcdcde;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        #fraud-check-result h4 {
            margin-top: 0;
            color: #1d2327;
            border-bottom: 1px solid #dcdcde;
            padding-bottom: 10px;
            font-size: 16px;
        }
        .courier-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
            margin-top: 15px;
        }
        .courier-item {
            background: white;
            padding: 15px;
            border-radius: 6px;
            border: 1px solid #dcdcde;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }
        .courier-header {
            display: flex;
            align-items: center;
            margin-bottom: 10px;
        }
        .courier-header img {
            width: 50px;
            height: auto;
            margin-right: 10px;
            border-radius: 4px;
        }
        .courier-stats {
            font-size: 14px;
        }
        .success-text {
            color: #00a32a;
        }
        .error-text {
            color: #d63638;
        }
        @media (max-width: 782px) {
            .courier-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>

    <div class="fraud-check-container">
        <button type="button" class="button button-primary" id="fraud-check-btn">
            <span class="dashicons dashicons-shield-alt" style="vertical-align: middle; margin-right: 5px;"></span>
            Check Fraud Status
        </button>
        <div id="fraud-check-result"></div>
    </div>

    <script>
        jQuery(document).ready(function($) {
            const phone = '<?php echo esc_js($cleaned); ?>';
            const fraudbd_nonce = '<?php echo esc_js( wp_create_nonce('fraudbd_nonce') ); ?>';

            function renderResult(res) {
                let html = '<h4>Fraud Check Summary</h4>';
                html += `<p><strong>Total Orders:</strong> ${res.totalSummary.total}</p>`;
                html += `<p><strong class="success-text">Successful Deliveries:</strong> ${res.totalSummary.success}</p>`;
                html += `<p><strong class="error-text">Cancelled Orders:</strong> ${res.totalSummary.cancel}</p>`;
                html += `<p><strong>Success Rate:</strong> <span class="success-text">${res.totalSummary.successRate}%</span></p>`;
                html += `<p><strong>Cancel Rate:</strong> <span class="error-text">${res.totalSummary.cancelRate}%</span></p>`;
                html += '<h4>Courier Performance</h4><div class="courier-grid">';

                for (const [courier, info] of Object.entries(res.Summaries)) {
                    html += `<div class="courier-item">
                                <div class="courier-header">
                                    <img src="${info.logo}" alt="${courier}" />
                                    <strong>${courier}</strong>
                                </div>
                                <div class="courier-stats">
                                    <p>Total: ${info.total}</p>
                                    <p class="success-text">Success: ${info.success}</p>
                                    <p class="error-text">Cancel: ${info.cancel}</p>
                                </div>
                            </div>`;
                }
                html += '</div>';
                $('#fraud-check-result').html(html);
            }

            function fetchFromAPI(force = false) {
                const $btn = $('#fraud-check-btn');
                $btn.prop('disabled', true).html('<span class="spinner is-active" style="float:none;margin:0 5px;"></span> Checking...');
                $('#fraud-check-result').html('<div class="fraud-alert"><span class="spinner is-active" style="float:none;margin-right:5px;"></span> Checking phone number...</div>');

                $.ajax({
                    url: 'https://fraudbd.com/api/check-courier-info',
                    method: 'POST',
                    headers: {
                        'api_key': '<?php echo esc_js($api_key); ?>',
                        'user_name': '<?php echo esc_js($username); ?>',
                        'password': '<?php echo esc_js($password); ?>',
                        'Content-Type': 'application/json'
                    },
                    data: JSON.stringify({ phone_number: phone }),
                    success: function (res) {
                        if (res.status && res.data) {
                            renderResult(res.data);

                            // Save cache
                            $.post(ajaxurl, {
                                action: 'fraudbd_save_cached_result',
                                phone: phone,
                                data: JSON.stringify(res.data),
                                 _wpnonce: fraudbd_nonce
                            });
                        } else {
                            $('#fraud-check-result').html('<div class="fraud-alert fraud-alert-error">' + (res.message || 'No data found.') + '</div>');
                        }
                    },
                    error: function () {
                        $('#fraud-check-result').html('<div class="fraud-alert fraud-alert-error">API request failed. Please try again.</div>');
                    },
                    complete: function() {
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-shield-alt" style="vertical-align: middle; margin-right: 5px;"></span> Check Fraud Status');
                    }
                });
            }

            // Auto-load cache
            $.post(ajaxurl, {
                action: 'fraudbd_get_cached_result',
                phone: phone,
                _wpnonce: fraudbd_nonce
            }, function(response) {
                if (response.success && response.data) {
                    renderResult(response.data);
                } else {
                    fetchFromAPI(); // fetch fresh if not found
                }
            });

            $('#fraud-check-btn').click(function () {
                fetchFromAPI(true); // force refresh
            });
        });

    </script>
    <?php
});

add_action('wp_ajax_fraudbd_get_cached_result', 'fraudbd_get_cached_result');
add_action('wp_ajax_fraudbd_save_cached_result', 'fraudbd_save_cached_result');

function fraudbd_get_cached_result() {
    // Verify nonce
    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'fraudbd_nonce')) {
        wp_send_json_error('Security check failed.');
    }

    // Validate presence
    if (!isset($_POST['phone'])) {
        wp_send_json_error('Missing phone number.');
    }

    // Sanitize phone properly
    $phone = sanitize_text_field(wp_unslash($_POST['phone']));

    $key = 'fraudbd_cache_' . $phone;
    $cached = get_transient($key);

    if ($cached) {
        wp_send_json_success($cached);
    } else {
        wp_send_json_error('No cached data.');
    }
}

function fraudbd_save_cached_result() {
    // Ensure expected fields exist before accessing
    if (
        !isset($_POST['_wpnonce']) ||
        !isset($_POST['phone']) ||
        !isset($_POST['data'])
    ) {
        wp_send_json_error('Missing required fields.');
    }

    // Unslash and sanitize inputs
    $nonce     = sanitize_text_field(wp_unslash($_POST['_wpnonce']));
    $phone     = sanitize_text_field(wp_unslash($_POST['phone']));
    $data_raw  = wp_unslash($_POST['data']);

    // Verify nonce
    if (!wp_verify_nonce($nonce, 'fraudbd_nonce')) {
        wp_send_json_error('Security check failed.');
    }

    // Validate phone
    if (empty($phone)) {
        wp_send_json_error('Invalid phone number.');
    }

    // Decode JSON safely
    $data = json_decode($data_raw, true);
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
        wp_send_json_error('Invalid JSON data.');
    }

    // (Optional) Recursively sanitize if storing untrusted user content
    function sanitize_recursive($input) {
        if (is_array($input)) {
            return array_map('sanitize_recursive', $input);
        }
        return is_string($input) ? sanitize_text_field($input) : $input;
    }

    $data = sanitize_recursive($data);

    // Store transient
    $key = 'fraudbd_cache_' . $phone;
    set_transient($key, $data, 6 * HOUR_IN_SECONDS);

    wp_send_json_success('Cached successfully');
}


